#
# Copyright (c) 2010, W T Schueller
# All rights reserved.
# 
# Redistribution and use in source and binary forms, with or without modification,
# are permitted provided that the following conditions are met:
# 
# * Redistributions of source code must retain the above copyright notice,
#   this list of conditions and the following disclaimer.
# * Redistributions in binary form must reproduce the above copyright notice,
#   this list of conditions and the following disclaimer in the documentation
#   and/or other materials provided with the distribution.
# * Neither the name of the author nor the names of the contributors
#   may be used to endorse or promote products derived from this software
#   without specific prior written permission.
# 
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
# LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
# ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#

##
# \mainpage
#
# Tcl2doxy is designed as an input filter for doxygen. It translates tcl code
# into something that doxygen can parse und create a documentation for.
#
# Tcl2doxy uses a scanner - parser - backend approach. The scanner and parser
# are generated using fickle and tackle respectively. The backend simply traverses
# the syntax tree and produces the output.
#
# see source code of function \ref help for usage.

##
# \file
#
# \author W T Schueller
#
# \brief Tcl2doxy main file
#

##
# \brief array with human readable tokens
array set token [ list ]
##
# \brief main programme
# \param argv arguments list
#
proc main argv {
    #
    # read parser and lexer sources and other sources
    set src_dir [ file dirname [ info script ] ]
    foreach src { "lexer.tcl" "parser.tcl" "parser.tab.tcl" "syntree.tcl" } {
        source [ file join $src_dir $src ]
    }
    # ok now we do real work
    # command line options
    set args $argv
    set filter "tcl2doxy"
    set scriptlevel 1
    while { [ llength $args ] > 0 } {
        switch -glob -- [ lindex $args 0 ] {
            "-h*" {
                help
                exit
            }
            "-l*" {
                set filter "lexer"
                set args [ lrange $args 1 end ]
            }
            "-p*" {
                set filter "parser"
                set args [ lrange $args 1 end ]
            }
            "-d*" {
                set filter "deparser"
                set args [ lrange $args 1 end ]
            }
            "--" {
                set args [ lrange $args 1 end ]
                break
            }
            "-scriptlevel" {
                set scriptlevel 0
                set args [ lrange $args 1 end ]
            }
            "-*" {
                help
                exit 1
            }
            default {
                break
            }
        }
    }
    # the remaining argument must be the input file
    if { [ llength $args ] != 1 } then {
        help
        exit 1
    }
    # try to read it
    set r [ catch { open [ lindex $args 0 ] r } fp ]
    if { $r } then {
        puts stderr $fp
        exit 1
    }
    set tcl [ read $fp ]
    close $fp
    #
    switch -exact -- $filter {
        "lexer" {
            yy::reset $tcl
            while { 1 } {
                set r [ yy::lex ]
                set v [ lindex $r 2 ]
                set v [ string map { \n \\n \t \\t } $v ]
                set s [ lindex $r 1 ]
                set t [ lindex $r 0 ]
                if { $t == 0 } then {
                    break
                }
                puts "$t:$v"
            }
        }
        "parser" {
            set tree [ yy::parse $tcl ]
            puts [ dumptree $tree ]
        }
        "deparser" {
            set tree [ yy::parse $tcl ]
            set out [ yy::deparse $tree ]
            puts -nonewline $out
            if { $out ne $tcl } then {
                puts "output differs from input"
                error "deparser: output differs from input"
            }
        }
        "tcl2doxy" {
            set tree [ yy::parse $tcl ]
            set output [ tree2doxy $tree $scriptlevel ]
            ##
            # \todo ugly hack for nice block comments
            regsub -all {\*/([ \t]*\n[ \t]*)/\*} $output {\1 *} output 
            puts -nonewline $output
        }
    }
}
##
# \brief output help information
#
proc help { } {
    puts "Usage: tcl2doxy \[options\] tclfile

    tclfile         tcl file to convert

Options:
    -help           print this help message and quit
    -lexer          run lexer only
    -parser         run lexer and parser only
    -deparser       recreate tcl from syntax tree
    -scriptlevel    do not suppress commands at script level
    --              end of options, use if file name starts with a dash"
    if { [ info exist ::REVISION ] } then {
        puts ""
        puts "($::REVISION, patchlevel [ info patchlevel ])"
    }
}
#
main $argv
exit
#
# end of script
#

